<?php

namespace App\Models;

use App\CentralLogics\Helpers;
use http\Env\Request;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use App\Scopes\ZoneScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\DB;

class Order extends Model
{
    use HasFactory;

    protected $casts = [
        'order_amount' => 'float',
        'coupon_discount_amount' => 'float',
        'total_tax_amount' => 'float',
        'store_discount_amount' => 'float',
        'delivery_address_id' => 'integer',
        'delivery_man_id' => 'integer',
        'delivery_charge' => 'float',
        'original_delivery_charge'=>'float',
        'user_id' => 'integer',
        'scheduled' => 'integer',
        'store_id' => 'integer',
        'details_count' => 'integer',
        'module_id' => 'integer',
        'zone_id' => 'integer',
        'dm_vehicle_id' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'receiver_details'=>'array',
        'dm_tips'=>'float',
        'distance'=>'float',
        'prescription_order' => 'boolean',
        'is_read' => 'boolean',
        'order_description' => 'string'
    ];

    protected $appends = ['module_type'];

    public function setDeliveryChargeAttribute($value)
    {
        $this->attributes['delivery_charge'] = round($value, 3);
    }

    public function details()
    {
        return $this->hasMany(OrderDetail::class);
    }

    public function whatsAppStore()
    {
        return $this->belongsTo(WhatsAppStore::class, 'whatsapp_market_id');
    }

    public function delivery_man()
    {
        return $this->belongsTo(DeliveryMan::class, 'delivery_man_id');
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function coupon()
    {
        return $this->belongsTo(Coupon::class, 'coupon_code', 'code');
    }

    public function store()
    {
        return $this->belongsTo(Store::class, 'store_id');
    }

    public function zone()
    {
        return $this->belongsTo(Zone::class, 'zone_id');
    }

    public function module()
    {
        return $this->belongsTo(Module::class);
    }

    public function delivery_history()
    {
        return $this->hasMany(DeliveryHistory::class, 'order_id');
    }

    public function dm_last_location()
    {
        // return $this->hasOne(DeliveryHistory::class, 'order_id')->latest();
        return $this->delivery_man->last_location();
    }

    public function transaction()
    {
        return $this->hasOne(OrderTransaction::class);
    }

    public function parcel_category()
    {
        return $this->belongsTo(ParcelCategory::class);
    }

    public function refund()
    {
        return $this->hasOne(Refund::class, 'order_id');
    }

    public function getModuleTypeAttribute()
    {
        return $this->module?$this->module->module_type:null;
    }

    public function scopeAccepteByDeliveryman($query)
    {
        return $query->where('order_status', 'accepted');
    }

    public function scopePreparing($query)
    {
        return $query->whereIn('order_status', ['confirmed','processing','handover']);
    }

    public function scopeModule($query, $module_id)
    {
        return $query->where('module_id', $module_id);
    }

    public function scopeOngoing($query)
    {
        return $query->whereIn('order_status', ['accepted','confirmed','processing','handover','picked_up']);
    }

    public function scopeItemOnTheWay($query)
    {
        return $query->where('order_status','picked_up');
    }

    public function scopePending($query)
    {
        return $query->where('order_status','pending');
    }

    public function scopeFailed($query)
    {
        return $query->where('order_status','failed');
    }

    public function scopeCanceled($query)
    {
        return $query->where('order_status','canceled');
    }

    public function scopeDelivered($query)
    {
        return $query->where('order_status','delivered');
    }

    public function scopeNotRefunded($query)
    {
        return $query->where(function($query){
            $query->whereNotIn('order_status', ['refunded']);
        });
    }

    public function scopeRefunded($query)
    {
        return $query->where('order_status','refunded');
    }
    public function scopeRefund_requested($query)
    {
        return $query->where('order_status','refund_requested');
    }

    public function scopeRefund_request_canceled($query)
    {
        return $query->where('order_status','refund_request_canceled');
    }

    public function scopeSearchingForDeliveryman($query)
    {
        return $query->whereNull('delivery_man_id')->whereIn('order_type' , ['delivery','parcel'])->whereNotIn('order_status',['delivered','failed','canceled', 'refund_requested','refund_request_canceled', 'refunded']);
    }

    public function scopeDelivery($query)
    {
        return $query->where('order_type', '=' , 'delivery');
    }

    public function scopeScheduled($query)
    {
        return $query->whereRaw('created_at <> schedule_at')->where('scheduled', '1');
    }

    public function scopeOrderScheduledIn($query, $interval)
    {
        return $query->where(function($query)use($interval){
            $query->whereRaw('created_at <> schedule_at')->where(function($q) use ($interval) {
            $q->whereBetween('schedule_at', [Carbon::now()->toDateTimeString(),Carbon::now()->addMinutes($interval)->toDateTimeString()]);
            })->orWhere('schedule_at','<',Carbon::now()->toDateTimeString());
        })->orWhereRaw('created_at = schedule_at');

    }


    public function scopeStoreOrder($query)
    {
        return $query->where(function($q){
            $q->where('order_type', 'take_away')->orWhere('order_type', 'delivery');
        });
    }

    public function scopeDmOrder($query)
    {
        return $query->where(function($q){
            $q->where('order_type', 'parcel')->orWhere('order_type', 'delivery');
        });
    }

    public function scopeParcelOrder($query)
    {
        return $query->where('order_type', 'parcel');
    }

    public function scopePos($query)
    {
        return $query->where('order_type', '=' , 'pos');
    }

    public function scopeNotpos($query)
    {
        return $query->where('order_type', '<>' , 'pos');
    }

    public function getCreatedAtAttribute($value)
    {
        return date('Y-m-d H:i:s',strtotime($value));
    }

    protected static function booted()
    {
        static::addGlobalScope(new ZoneScope);
    }

    public static function getOriginalDeliveryCharge($deliveryCharge, $deliveryManId, $orderType = 'Parcel')
    {
        $DM = DeliveryMan::where('id', $deliveryManId)->first();
        $isFreeLancer = $DM->earning == 1;
        if ($isFreeLancer){
            if(DeliveryMan::isHaveActiveMonthlySubscription($deliveryManId)){
                $originalDeliveryCharge = $deliveryCharge;
            }else {
                if ($orderType == 'WhatsApp') {
                    $deliveryManCommission = BusinessSetting::where('key', 'whatsapp_commission_dm')->first()->value;
                } else {
                    $deliveryManCommission = 100 - BusinessSetting::where('key', 'admin_commission')->first()->value;
                }
                $originalDeliveryCharge = ($deliveryCharge * $deliveryManCommission) / 100; // Delivery Man Commission
            }
        }else{
            $originalDeliveryCharge = 0;
        }
        return $originalDeliveryCharge;
    }

    public static function sendSystemMessages($orderId)
    {
        $order = Order::where('id', $orderId)->first();
        if(!$order or !isset($order->delivery_man_id)){
            return false;
        }
        $dm = $order->delivery_man;

        $services = [
            'Parcel' => 'توصيل طرد',
            'Restaurants' => 'مطاعم',
            'Cafe' => 'قهوة',
            'Sweets' => 'حلويات',
            'Juice' => 'عصير',
            'Pharmacy' => 'أدوية',
        ];

        if($order->module->module_name != 'Parcel') {
            $service_name = $services[$order->module->module_name] ?? '';
        }else{
            $translation = DB::table('translations')->where('translationable_type', 'App\Models\ParcelCategory')->where('translationable_id', $order->parcel_category_id)->where('key', 'name')->where('locale', 'ar')->first();
            $service_name = $translation->value ?? '';
        }
        $order_amount = $order->order_type != 'parcel' ? $order->order_amount : 0;
        $delivery_charge = $order->delivery_charge;
        $total_tax_amount = $order->total_tax_amount;
        $coupon_discount_amount_for_delivery = 0;
        $coupon_discount_amount_for_order_amount = 0;
        if($order->coupon_code != null){
            $couponInfo = Coupon::where('code', $order->coupon_code)->first();
            if($couponInfo->coupon_type == 'store_wise'){
                $coupon_discount_amount_for_order_amount = $order->coupon_discount_amount;
            }else if($couponInfo->coupon_type == 'vip'){
                $coupon_discount_amount_for_order_amount = $order->coupon_discount_amount - $delivery_charge;
                $coupon_discount_amount_for_delivery = $delivery_charge;
            }else{
                $coupon_discount_amount_for_delivery = $order->coupon_discount_amount;
            }
        }
        $totalAmountRequiredFromCustomer = ($order_amount + $delivery_charge + $total_tax_amount) - ($coupon_discount_amount_for_delivery + $coupon_discount_amount_for_order_amount);
        if(isset($order->store)) {
            $totalAmountForRestaurant = $order_amount - ($order_amount * $order->store->comission / 100);
        }
        $message1 = '<strong>تفاصيل الطلب</strong><br />';
        if(!empty($service_name)) {
            $message1 .= 'اسم الخدمة: ' . '<strong>' . $service_name . '</strong><br />';
        }
        if(isset($order->store)){
            $message1 .= 'اسم المتجر: ' . '<strong>'. $order->store->name .'</strong><br />';
        }

        $message1 .= 'المسافة: ' . '<strong>'. $order->distance .'</strong> كم<br />';

        if($order->order_type != 'parcel'){
            $message1 .= 'قيمة الطلب: ' . '<strong>'. number_format($order_amount) .'</strong> ل.س' . '<br />';
        }

        $message1 .= 'قيمة التوصيل: ' . '<strong>'. number_format($delivery_charge) .'</strong> ل.س' . '<br />';

        if($order->order_type != 'parcel') {
            $message1 .= 'الضريبة: ' . '<strong>' . number_format($total_tax_amount) . '</strong> ل.س' . '<br />';
        }

        if($order->order_type != 'parcel' && $coupon_discount_amount_for_order_amount > 0) {
            $message1 .= 'خصم الكوبون على قيمة الطلب: ' . '<strong>' . number_format($coupon_discount_amount_for_order_amount) . '</strong> ل.س' . '<br />';
        }
        //if($coupon_discount_amount_for_delivery > 0) {
            $message1 .= 'خصم الكوبون على أجرة التوصيل: ' . '<strong>' . number_format($coupon_discount_amount_for_delivery) . '</strong> ل.س' . '<br />';
        //}

        if($coupon_discount_amount_for_delivery > 0 or $coupon_discount_amount_for_order_amount > 0){
            $message1 .= 'لا تقلق.. سيتم تعويض المندوب بالحسم على حسابه.' . '<br />';
        }

        if($order->order_type != 'parcel') {
            $message1 .= 'المبلغ الإجمالي المطلوب استلامه من الزبون: ' . '<strong>' . number_format($totalAmountRequiredFromCustomer) . '</strong> ل.س' . '<br />';
        }else{
            //if($coupon_discount_amount_for_delivery > 0){
                $message1 .= 'قيمة التوصيل بعد الخصم: ' . '<strong>'. number_format($delivery_charge - $coupon_discount_amount_for_delivery) .'</strong> ل.س' . '<br />';
            //}
        }

        $message2 = '';
        if($order->order_type != 'parcel'){
            if($order->details->count() > 0){
                $message2 .= '<strong>محتويات الطلب: </strong><br />';
                foreach ($order->details as $item){
                    $message2 .= json_decode($item->item_details)->name . ' العدد: ' . '('. $item->quantity .') - ' . ' السعر الإجمالي: ' . number_format(($item->price - $item->discount_on_item) * $item->quantity) . ' ل.س';
                    if($item->add_ons != '[]'){
                        $add_ons = json_decode($item->add_ons);
                        $message2 .= '<br />الإضافات: <br />';
                        foreach ($add_ons as $add_on){
                            $message2 .= $add_on->name . ' العدد: ' . " (". $add_on->quantity .") - ". ' السعر الإجمالي: ' . number_format($add_on->price * $add_on->quantity) . ' ل.س' ."<br />";
                        }
                    }
                    $message2 .= '<br />';
                }
                $message2 .= '<br />';
            }
        }

        if(!empty($order->order_description) and $order->order_type == 'parcel') {
            $message2 .= 'وصف الطلب: ' . '<br />';
            $message2 .= $order->order_description;
        }

        if(!empty($order->order_note)){
            $message2 .= 'ملاحظات اضافية: ' . '<br />';
            $message2 .= $order->order_note;
        }

        if(isset($order->store)) {
            $message3 = 'المبلغ الواجب دفعه للمطعم: ' . '<strong>' . number_format($totalAmountForRestaurant + $total_tax_amount) . ' ل.س' . '</strong><br /><br />';
            if(isset($couponInfo)){
                if($couponInfo->coupon_type == 'store_wise'){
                    $result = $totalAmountRequiredFromCustomer - $totalAmountForRestaurant;
                    if($result < $order->original_delivery_charge) {
                        $compensationValue = $order->original_delivery_charge - $result;
                    }
                }else if($couponInfo->coupon_type == 'free_delivery'){
                    $compensationValue = $order->original_delivery_charge;
                }
                if(isset($compensationValue) and $compensationValue > 0){
                    $message3 = 'لا تقلق سيتم تعويضك بمبلغ : ' . '<strong>' . number_format($compensationValue) . ' ل.س على حسابك' . '</strong><br />';
                }
            }
        }
        $message4 = 'مرحباً..' . '<br />';
        $message4 .= 'معك المندوب: ' . '<strong>'. $dm->f_name . ' ' . $dm->l_name .'</strong><br />';
        $message4 .= 'سأكون سعيد بخدمتك..' . '<br />';

        $message1 = str_replace('<strong>', '', $message1);
        $message1 = str_replace('</strong>', '', $message1);
        $message1 = str_replace('<br />', '
', $message1); //Replace with new line

        if(isset($message2)) {
            $message2 = str_replace('<strong>', '', $message2);
            $message2 = str_replace('</strong>', '', $message2);
            $message2 = str_replace('<br />', '
', $message2); //Replace with new line
        }

        if(isset($message3)) {
            $message3 = str_replace('<strong>', '', $message3);
            $message3 = str_replace('</strong>', '', $message3);
            $message3 = str_replace('<br />', '
', $message3); //Replace with new line
        }
        $message4 = str_replace('<strong>', '', $message4);
        $message4 = str_replace('</strong>', '', $message4);
        $message4 = str_replace('<br />', '
', $message4); //Replace with new line

        $sender = UserInfo::where('deliveryman_id', $dm->id)->first();
        if (!$sender) {
            $sender = new UserInfo();
            $sender->deliveryman_id = $dm->id;
            $sender->f_name = $dm->f_name;
            $sender->l_name = $dm->l_name;
            $sender->phone = $dm->phone;
            $sender->email = $dm->email;
            $sender->image = $dm->image;
            $sender->save();
        }else{
            $sender->image = $dm->image;
            $sender->save();
        }

        $receiver = UserInfo::where('user_id', $order->user_id)->first();
        $user = User::find($order->user_id);
        $fcm_token = $user->cm_firebase_token;

        if (!$receiver) {
            $receiver = new UserInfo();
            $receiver->user_id = $user->id;
            $receiver->f_name = $user->f_name;
            $receiver->l_name = $user->l_name;
            $receiver->phone = $user->phone;
            $receiver->email = $user->email;
            $receiver->image = $user->image;
            $receiver->save();
        }else{
            $receiver->image = $user->image;
            $receiver->save();
        }

        $conversation = Conversation::where(function ($query) use($sender, $receiver) {
            $query->where('sender_id', $sender->id)->where('sender_type', 'delivery_man')->where('receiver_id', $receiver->id)->where('receiver_type', 'customer');
        })->orWhere(function ($query) use($sender, $receiver) {
            $query->where('receiver_id', $sender->id)->where('receiver_type', 'delivery_man')->where('sender_id', $receiver->id)->where('sender_type', 'customer');
        })->first();

        if(!$conversation) {
            $conversation = Conversation::create([
                'sender_id' => $sender->id,
                'sender_type' => 'delivery_man',
                'receiver_id' => $receiver->id,
                'receiver_type' => 'customer',
                'last_message_id' => null,
                'last_message_time' => now(),
                'unread_message_count' => 0,
            ]);
        }

        $conversationId = $conversation->id;

        $firstMessage = new Message();
        $firstMessage->conversation_id = $conversationId;
        $firstMessage->order_id = $order->id;
        $firstMessage->sender_id = $sender->id;
        $firstMessage->message = $message1;
        $firstMessage->file = null;
        $firstMessage->is_visible_to_dm = 1;
        $firstMessage->is_visible_to_customer = 1;
        $firstMessage->save();

        if(isset($message2)) {
            $secondMessage = new Message();
            $secondMessage->conversation_id = $conversationId;
            $secondMessage->order_id = $order->id;
            $secondMessage->sender_id = $sender->id;
            $secondMessage->message = $message2;
            $secondMessage->file = null;
            $secondMessage->is_visible_to_dm = 1;
            $secondMessage->is_visible_to_customer = 1;
            $secondMessage->save();
        }

        if(isset($message3)) {
            $thirdMessage = new Message();
            $thirdMessage->conversation_id = $conversationId;
            $thirdMessage->order_id = $order->id;
            $thirdMessage->sender_id = $sender->id;
            $thirdMessage->message = $message3;
            $thirdMessage->file = null;
            $thirdMessage->is_visible_to_dm = 1;
            $thirdMessage->is_visible_to_customer = 0;
            $thirdMessage->save();
        }

        $forthMessage = new Message();
        $forthMessage->conversation_id = $conversationId;
        $forthMessage->order_id = $order->id;
        $forthMessage->sender_id = $sender->id;
        $forthMessage->message = $message4;
        $forthMessage->file = null;
        $forthMessage->is_visible_to_dm = 1;
        $forthMessage->is_visible_to_customer = 1;
        $forthMessage->save();

        $unread_message_count = 3;
        if(isset($message3)) $unread_message_count = 4;

        $conversation->unread_message_count = $conversation->unread_message_count ? $conversation->unread_message_count + $unread_message_count : $unread_message_count;
        $conversation->last_message_id = $forthMessage->id;
        $conversation->last_message_time = $forthMessage->created_at;
        $conversation->save();

        $data = [
            'title' => translate('messages.Message'),
            'description' => translate('messages.you_have_received_new_message'),
            'order_id' => $order->id,
            'image' => '',
            'message' => json_encode($message4),
            'type' => 'message',
            'conversation_id' => $conversationId,
            'sender_type' => 'delivery_man'
        ];
        Helpers::send_push_notif_to_device($fcm_token, $data);

        return $conversationId;
    }

    public static function generateOrderId()
    {
        $countOfOrders = Order::count();
        if($countOfOrders == 0) return 100000 + 1;

        $getMaxId = Order::select('id')->orderBy('id', 'DESC')->first();
        $maxId = $getMaxId->id + 1;

        if (Order::find($maxId)) {
            $maxId = $maxId + 1;
        }
        return $maxId;
    }

    public static function sendInvoiceMessage($orderId, $conversationId)
    {
        $order = Order::where('id', $orderId)->first();
        if(!$order or !isset($order->delivery_man_id)){
            return false;
        }
        $dm = $order->delivery_man;
        $total_invoice = $order->order_type == 'parcel' ? $order->total_invoice : $order->order_amount;
        $delivery_charge = $order->delivery_charge;
        $tax = $order->total_tax_amount;
        $coupon = $order->coupon_discount_amount;
        $total = $delivery_charge + $total_invoice + $tax - $coupon;

        $message1 = '<strong>قام المندوب بإصدار فاتورة</strong><br />';
        $message1 .= 'قيمة المشتريات: ' . '<strong>'. number_format($total_invoice) .'</strong> ل.س<br />';
        $message1 .= 'أجور التوصيل: ' . '<strong>'. number_format($delivery_charge) .'</strong> ل.س<br />';
        if($order->order_type != 'parcel') {
            $message1 .= 'قيمة الضريبة: ' . '<strong>' . number_format($tax) . '</strong> ل.س<br />';
        }
        $message1 .= 'قيمة الكوبون: ' . '<strong>'. number_format($coupon) .'</strong> ل.س<br />';
        $message1 .= 'المبلغ الإجمالي: ' . '<strong>'. number_format($total) .'</strong> ل.س<br />';

        $message1 = str_replace('<strong>', '', $message1);
        $message1 = str_replace('</strong>', '', $message1);
        $message1 = str_replace('<br />', '
', $message1); //Replace with new line

        $user = User::find($order->user_id);
        $conversation = Conversation::where('id', $conversationId)->first();

        $deliveryManSenderId = UserInfo::where('deliveryman_id', $dm['id'])->first()->id;

        $firstMessage = new Message();
        $firstMessage->conversation_id = $conversationId;
        $firstMessage->order_id = $order->id;
        $firstMessage->sender_id = $deliveryManSenderId;
        $firstMessage->message = $message1;
        $firstMessage->file = null;
        $firstMessage->is_visible_to_dm = 1;
        $firstMessage->is_visible_to_customer = 1;
        $firstMessage->save();

        $conversation->unread_message_count = $conversation->unread_message_count ? $conversation->unread_message_count + 2 : 2;
        $conversation->last_message_id = $firstMessage->id;
        $conversation->last_message_time = $firstMessage->created_at;
        $conversation->save();

        $data = [
            'title' => translate('messages.Message'),
            'description' => translate('messages.you_have_received_new_message'),
            'order_id' => $order->id,
            'image' => '',
            'message' => json_encode($message1),
            'type' => 'message',
            'conversation_id' => $conversationId,
            'sender_type' => 'delivery_man'
        ];
        Helpers::send_push_notif_to_device($user->cm_firebase_token, $data);

        return $conversationId;
    }

    public static function compensateDeliveryManForDistance($order)
    {
        $distance_between_dm_and_source_location = $order->distance_between_dm_and_source_location;
        if(empty($distance_between_dm_and_source_location) or $distance_between_dm_and_source_location == 0){
            return;
        }
        $max_km_to_compensate_the_dm_for_the_distance = \App\Models\BusinessSetting::where('key', 'max_km_to_compensate_the_dm_for_the_distance')->first()->value;
        $amount_to_compensate_the_dm_for_the_distance_in_one_km = \App\Models\BusinessSetting::where('key', 'amount_to_compensate_the_dm_for_the_distance_in_one_km')->first()->value;
        if($max_km_to_compensate_the_dm_for_the_distance == 0 or $amount_to_compensate_the_dm_for_the_distance_in_one_km == 0){
            return;
        }
        if($distance_between_dm_and_source_location > $max_km_to_compensate_the_dm_for_the_distance){
            $distance_between_dm_and_source_location = $max_km_to_compensate_the_dm_for_the_distance;
        }

        $amount = $distance_between_dm_and_source_location * $amount_to_compensate_the_dm_for_the_distance_in_one_km;
        if($amount == 0){
            return;
        }

        $dm = DeliveryMan::findOrFail($order->delivery_man_id);
        $current_balance = $dm->wallet ? $dm->wallet->collected_cash : 0;

        $account_transaction = new AccountTransaction();
        $account_transaction->from_type = 'deliveryman';
        $account_transaction->from_id = $order->delivery_man_id;
        $account_transaction->method = 'By system';
        $account_transaction->ref = 'تم تعويض المندوب عن المسافة بينه وبين مكان استلام الطلب رقم: ' . $order->id;
        $account_transaction->amount = (float)$amount;
        $account_transaction->current_balance = $current_balance;
        $account_transaction->transaction_date = now();
        $account_transaction->save();

        $getDMLogMessage = Helpers::getDMLogMessage('compensate_dm_for_distance');
        $getDMLogMessage = str_replace('******', $order->id, $getDMLogMessage);
        DMLog::createLog($order->delivery_man_id, $getDMLogMessage, $amount, null);
    }

    public static function calcOrderItemsAmount($order): array
    {
        $orderDetails = OrderDetail::where('order_id', $order->id)->get();
        $store = Store::where('id', $order->store_id)->first();
        $itemsPrice = 0;
        foreach ($orderDetails as $item){
            $itemsPrice += (($item->price - $item->discount_on_item) * $item->quantity);
            $addons = json_decode($item->add_ons);
            if(count($addons) > 0){
                foreach ($addons as $addon){
                    $addonPrice = $addon->price;
                    $addonQuantity = $addon->quantity;
                    $itemsPrice += ($addonPrice * $addonQuantity);
                }
            }
        }
        $taxPercent = $store->tax;
        $commissionPercent = $store->comission;
        $tax = ($itemsPrice * $taxPercent) / 100;
        $adminCommission = ($itemsPrice * $commissionPercent) / 100;
        $storeAmount = $itemsPrice - $adminCommission;

        return [
            'itemsPrice' => $itemsPrice,
            'tax' => $tax,
            'storeAmount' => $storeAmount,
            'adminCommission' => $adminCommission,
            'total' => $itemsPrice + $tax,
        ];
    }

    public static function editOrderMenu($request)
    {
        $cartItem = OrderDetail::findOrFail($request->order_details_id);
        //OrderDetail::where('id', $request->order_details_id)->update(['quantity' => $request['quantity']]);

        $payload = [];
        $totalAddOnPrice = 0;
        if(!isset($request->addons_array)) {
            $ids = $request->input('addon_id', []);
            $addons = AddOn::withoutGlobalScope(\App\Scopes\StoreScope::class)
                ->active()
                ->whereIn('id', $ids)
                ->get()
                ->keyBy('id');
            foreach ($ids as $id) {
                if (!isset($addons[$id])) {
                    continue;
                }
                $qtyTxt = 'addon-quantity' . $id;
                $qty = $request->$qtyTxt;

                $addon = $addons[$id];

                if($qty > 0) {
                    $totalAddOnPrice += (float)($addon->price * $qty);
                    $payload[] = [
                        'id' => (int)$addon->id,
                        'name' => (string)$addon->name,
                        'price' => (float)$addon->price,
                        'quantity' => $qty,
                    ];
                }
            }
        }else{
            $payload = $request->addons_array;
        }

        $cartItem->quantity = $request['quantity'];
        $cartItem->add_ons = json_encode($payload);
        $cartItem->total_add_on_price = $totalAddOnPrice;
        $cartItem->save();

        Order::updateOrderPrice($cartItem->order_id);
    }

    public static function updateOrderPrice($orderId)
    {
        $order = Order::where('id', $orderId)->first();
        $calcOrderItemsPrice = Order::calcOrderItemsAmount($order);
        $order->order_amount = $calcOrderItemsPrice['itemsPrice'];
        $order->total_tax_amount = $calcOrderItemsPrice['tax'];
        $order->save();

        OrderTransaction::where('order_id', $order->id)->update([
            'order_amount' => $calcOrderItemsPrice['itemsPrice'],
            'store_amount' => $calcOrderItemsPrice['storeAmount'],
            'admin_commission' => $calcOrderItemsPrice['adminCommission'],
            'tax' => $calcOrderItemsPrice['tax'],
        ]);
    }

}
