<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ApiSubscription;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ApiSubscriptionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\Response
     */
    public function index()
    {
        $apiSubscriptions = ApiSubscription::orderBy('id', 'DESC')->paginate(20);
        return view('admin-views.api-subscriptions.index', compact('apiSubscriptions'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin-views.api-subscriptions.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request): \Illuminate\Http\RedirectResponse
    {
        $validated = $request->validate([
            'name'       => 'required|string|max:255',
            'address'    => 'nullable|string|max:255',
            'phone'      => 'required|string|max:20',
            'email'      => 'required|email|unique:api_subscriptions,email',
            'start_date' => 'required|date',
            'end_date'   => 'required|date|after_or_equal:start_date',
            'get_items'  => 'required|boolean',
            'delivery'   => 'required|boolean',
            'status'   => 'required|boolean',
            'server_link_for_notification'   => 'nullable',
            'server_api_key'   => 'nullable',
            'server_secret_key'   => 'nullable',
        ]);

        do {
            $apiKey = Str::random(64);
        } while (ApiSubscription::where('api_key', $apiKey)->exists());

        do {
            $secretKey = Str::random(32);
        } while (ApiSubscription::where('secret_key', $secretKey)->exists());

        ApiSubscription::create([
            'name'        => $validated['name'],
            'address'     => $validated['address'],
            'phone'       => $validated['phone'],
            'email'       => $validated['email'],
            'start_date'  => $validated['start_date'],
            'end_date'    => $validated['end_date'],
            'get_items'   => $validated['get_items'],
            'delivery'    => $validated['delivery'],
            'api_key'     => $apiKey,
            'secret_key'  => $secretKey,
            'status'  => $validated['status'],
            //'server_link_for_notification'  => $validated['server_link_for_notification'],
            //'server_api_key'  => $validated['server_api_key'],
            //'server_secret_key'  => $validated['server_secret_key'],
        ]);

        Toastr::success('Subscription created successfully.');
        return redirect()->back();
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\ApiSubscription  $apiSubscription
     * @return \Illuminate\Http\Response
     */
    public function show(ApiSubscription $apiSubscription)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\ApiSubscription  $apiSubscription
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View|\Illuminate\Http\Response
     */
    public function edit(ApiSubscription $apiSubscription)
    {
        return view('admin-views.api-subscriptions.edit', compact('apiSubscription'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\ApiSubscription  $apiSubscription
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, ApiSubscription $apiSubscription): \Illuminate\Http\RedirectResponse
    {
        $validated = $request->validate([
            'name'       => 'required|string|max:255',
            'address'    => 'nullable|string|max:255',
            'phone'      => 'required|string|max:20',
            'email'      => 'required|email|unique:api_subscriptions,email,' . $apiSubscription->id,
            'start_date' => 'required|date',
            'end_date'   => 'required|date|after_or_equal:start_date',
            'get_items'  => 'required|boolean',
            'delivery'   => 'required|boolean',
            'status'   => 'required|boolean',
            'server_link_for_notification'   => 'nullable',
            'server_api_key'   => 'nullable',
            'server_secret_key'   => 'nullable',
        ]);

        $apiSubscription->update($validated);

        Toastr::success('Subscription saved successfully.');
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\ApiSubscription  $apiSubscription
     * @return \Illuminate\Http\Response
     */
    public function destroy(ApiSubscription $apiSubscription)
    {
        //
    }
}
